module axis_m
#( parameter
    SIZE_S          = 3'b010        // always refer to DATW_S
,   DATW_ST         =  64
,   KEPW_ST         = (DATW_ST/8)   // always refer to DATW_ST
,   STBW_S          = (DATW_ST/8)
,   DTMP            =  4096
,   STMP            =  DTMP/KEPW_ST   
)
(
    input                       S_AXI_ACLK
,   input                       S_AXI_ARESETN

    // AXI Stream Master:  m_axis
,   output reg [   DATW_ST-1:0] m_axis_tdata
,   output reg [   KEPW_ST-1:0] m_axis_tkeep
,   output reg                  m_axis_tlast
,   output reg                  m_axis_tvalid
,   input                       m_axis_tready
,   input byte                  i_req
,   input byte                  i_data[0:DTMP-1]
,   input int                   i_len
,   input int                   i_last_keep

    //==================================
    // Unused Signals
    //==================================
);

    //---------------------------------------------------------------------
    // Constant Declarations
    //---------------------------------------------------------------------

    localparam IDLE = 0;
    localparam DATA = 1;

    //---------------------------------------------------------------------
    // Reg/Wire
    //---------------------------------------------------------------------

    reg        [           1:0] r_state                 , n_state                   ;
    integer                     r_len                   , n_len                     ;
    integer                     r_count                 , n_count                   ;
    byte                        r_data [0:DTMP-1]       , n_data [0:DTMP-1]         ;
    reg                         r_tvalid                , n_tvalid                  ;
    reg        [   DATW_ST-1:0] r_tdata                 , n_tdata                   ;
    reg        [   KEPW_ST-1:0] r_tkeep                 , n_tkeep                   ;
    reg        [   KEPW_ST-1:0] r_last_keep             , n_last_keep               ;
    reg                         r_tlast                 , n_tlast                   ;

    //---------------------------------------------------------------------
    // Instance
    //---------------------------------------------------------------------

    //---------------------------------------------------------------------
    // Main
    //---------------------------------------------------------------------

    //------------------------------------------------------
    // Combo
    //------------------------------------------------------

    integer i;

    always@(*) 
    begin: COMBO_RECV
        n_state = r_state;
        n_tlast = 0;
        n_tkeep = 8'h0;
        case(r_state)
            IDLE: begin
                n_tvalid = 0;
                if (i_req && m_axis_tready) begin
                    n_last_keep = i_last_keep;
                    n_len = i_len;
                    n_data = i_data;  
                    n_count = 0;
                    n_state = DATA;
                end
            end
            DATA: begin
                n_tvalid = 1;
                for ( i = 0; i < KEPW_ST; i = i + 1 )
                    n_tdata[i*KEPW_ST+:8] = r_data[(r_count*KEPW_ST)+i];
                if (r_count == r_len) begin
                    n_tkeep = r_last_keep;
                    n_tlast = 1;
                end else begin 
                    n_tkeep = 8'hff;
                end
                if (m_axis_tready) begin
                    n_count = r_count + 1;
                    if (r_count == r_len) begin
                        n_state = IDLE; 
                    end
                end
            end
        endcase
    end
    
    //------------------------------------------------------
    // FF
    //------------------------------------------------------
    int j;
    always@(posedge S_AXI_ACLK)
    begin: FF_SEND
        if (~S_AXI_ARESETN) begin
            for (j = 0; j < DTMP; j++)
                r_data[i] <= 0;
        end else begin
            r_data <= n_data;
        end
        r_state  <= (~S_AXI_ARESETN) ? IDLE : n_state  ;
        r_len    <= (~S_AXI_ARESETN) ? 0 : n_len    ;
        r_count  <= (~S_AXI_ARESETN) ? 0 : n_count  ;
        r_tvalid <= (~S_AXI_ARESETN) ? 0 : n_tvalid ;
        r_tdata  <= (~S_AXI_ARESETN) ? 0 : n_tdata  ;
        r_tkeep  <= (~S_AXI_ARESETN) ? 0 : n_tkeep  ;
        r_tlast  <= (~S_AXI_ARESETN) ? 0 : n_tlast  ;
        r_last_keep <= (~S_AXI_ARESETN) ? 0 : n_last_keep;
    end // FF_SEND
    
    //---------------------------------------------------------------------
    // Primary Output Assignment
    //---------------------------------------------------------------------

    // Combo Out
    // Const Out
    // Reg   Out
    assign m_axis_tvalid = r_tvalid;
    assign m_axis_tdata  = r_tdata;
    assign m_axis_tkeep  = r_tkeep;
    assign m_axis_tlast  = r_tlast;

    initial begin
        r_state  <= 0; 
        r_len    <= 0;
        r_count  <= 0; 
        r_tvalid <= 0; 
        r_tdata  <= 0; 
        r_tkeep  <= 0; 
        r_tlast  <= 0; 
        r_last_keep <= 0;
    end
endmodule
